/*
 * File:   main.c
 * Author: pproser
 *
 * Created on July 24, 2022, 5:44 PM
 * V1.0 09 Sep 2022 fully functional
 * V1.2 11 Sep 2022 rewrote IR Receiver to state machine
 * 
 * Default = Philips RC-5 TV
 * Power on with pins 3-4 on prog header short = Philips RC-5 RECEIVER
 * Power on with pins 3-5 on prog header short = philips RC-5 TV
 * Then power on with no shorts on this 
 * 
 */

#include "config.h"
#include "io.h"
#include "flash.h"
#include "softi2c.h"  // not imported yet
#include "util.h"  //not imported yet
#include <xc.h>


/*******************************************************************************
  Function:
    void Volume_Initialize ( void )

  Remarks:
    See prototype in util.h.
******************************************************************************/
void Volume_Initialize ( void )
{
    /* Place the App state machine in its initial state. */
    Volume_Data.state           = VOLUME_STATE_INIT;
    Volume_Data.RC5_state       = Out_1;
    Volume_Data.Volume          = VOLUME_VAL_LOWER;
    Volume_Data.Timer_Time      = 0;
    Volume_Data.Timer_Temp      = 0;
    Volume_Data.Interval        = 0;
    Volume_Data.Sample_Time     = 0;
    Volume_Data.ROT_State       = 0;
    Volume_Data.ROT_State_Prev  = Phase0;
    Volume_Data.IR_State        = Phase0;
    Volume_Data.IR_State_Prev   = 0;   
    Volume_Data.Valid_Input     = 0;   
    Volume_Data.Action          = No_Action;   
    Volume_Data.Increment       = 1;
    Volume_Data.Slow_Loops      = 0; 
    Volume_Data.Fast_Loops      = 0; 
    Volume_Data.TimeOut         = 0; 
    Volume_Data.Toggle          = 0;
    Volume_Data.Address         = 0;
    Volume_Data.Command         = 0;
    Volume_Data.Save_Timer      = 0;
    Volume_Data.Data_To_Save    = 0;
    Volume_Data.Mute            = 0;
    Volume_Data.Source          = 1;
    Volume_Data.IR_Addr         = Volume_Address_IR;
    Volume_Data.IR_RX           =0;
}


void main(void) {
char Volume_Count; 
char Temp_PortA, Current_PortA;    
char Bit_Count=0;

    // initialise the data structures and state machine
    Volume_Initialize();
    __delay_ms(startup_delay);
    /* Check the application's current state. */
    do{
        switch ( Volume_Data.state )
        {
            /* Application's initial state. */
            case VOLUME_STATE_INIT:
            {
                // some configuration stuff to get things rolling 
                WDTCON=0;       //off under software control
                OSCFRQ=2;       //4MHz - this should be fine for this application
                //Check what sort of remote is connected
                //RA2 = Test this input and see if it is pulled low for TV Channel
                //RA1 = Test this input and see if it is pulled low for RCVR channel    //RA2 = Test this input and see if it is pulled low for TV Channel
                Io_Check_Remote();
                __delay_ms(settle_time_IR_Read);  /* let the port settle */
                Current_PortA = PORTA & REM_BITS;    /* Masks used PORTA input bits */
                loadRemCodeFromFlash();    /* look what is in FASH */
                /* if FLASH has Check_Bytes 0,1,2 = Tgm then TGM has been here already */
                //Initialise the PIC I/O 
                ioinit();
                //set up the soft I2C and set outputs in a sane initial state
                I2Cstop();
                if ((Volume_Data.Check_Byte_0 ==  Flash_Init_Byte_0) && (Volume_Data.Check_Byte_1 ==  Flash_Init_Byte_1) && (Volume_Data.Check_Byte_2 ==  Flash_Init_Byte_2))
                {
                    if(Current_PortA == (REM_BITS - TV_Port))   /* i.e. TV port is low */
                    {
                        Volume_Data.Source = 1; /* TV */
                        Volume_Data.IR_Addr = Volume_Address_IR;
                        SaveVolToFlash(); /* was changed - needs saving */
                    }
                    else if (Current_PortA == (REM_BITS - RCVR_Port)) /* i.e. SAY bit is low */
                    {
                        Volume_Data.Source = 0; /* RCVR */
                        Volume_Data.IR_Addr = Volume_Address_IR_RCVR;
                        SaveVolToFlash(); /* was changed - needs saving */
                    }
                    else
                    {
                    }; /* leave it as it is*/
                }
                else /* we are pretty sure things are not initialised */
                {
                    if(Current_PortA == (REM_BITS - TV_Port))   /* i.e. TV Port is low */
                        Volume_Data.Source = 1; /* TV */
                    else if (Current_PortA == (REM_BITS - RCVR_Port)) /* i.e. RCVR port is low */
                        Volume_Data.Source = 0; /* RCVR */
                    else
                        Volume_Data.Source = 1; /* default to TV */
                    Volume_Data.Volume = VOLUME_DEFAULT;  /* default this */
                    Volume_Data.Check_Byte_0 = Flash_Init_Byte_0;
                    Volume_Data.Check_Byte_1 = Flash_Init_Byte_1;
                    Volume_Data.Check_Byte_2 = Flash_Init_Byte_2;
                    SaveVolToFlash(); /* was blank - needs saving */
                }
                
                if(Volume_Data.Source == 1) /* TV */
                    Volume_Data.IR_Addr = Volume_Address_IR;
                else
                    Volume_Data.IR_Addr = Volume_Address_IR_RCVR;

                //now read the EEPORM for the stored value
                loadVolFromFlash();
                // set up timer:  1 us clock - Read TMR0L and TMR0H
                T0CON0bits.OUTPS = 0; // 1:1 prescaler
                T0CON0bits.MD16  = 1; // 16 bit timer
                T0CON0bits.EN = 1;    // start
                T0CON1bits.CKPS = 0;  // 1:1 clock prescale
                T0CON1bits.ASYNC = 0; // synchronise to sys FOSC/4
                T0CON1bits.CS = 2;    // 1:1 clock prescale

                // Ramp volume from zero
                for (Volume_Count = VOLUME_VAL_LOWER; Volume_Count <= Volume_Data.Volume; Volume_Count++)
                {
                    I2C_Volume_Word(Volume_Count);
                    __delay_ms(Volume_Delay_Ms);
                };
                /* now to the idle state */
                Volume_Data.state = VOLUME_STATE_IDLE;
                Current_PortA = PORTA & PORT_BITS;    /* Masks used PORTA input bits */
                Temp_PortA = Current_PortA;
                
                /* check to see if the user wants to change the IR Address between TV1 and TV2 */
                
            }
            break;

            /* Application's idle state.                                                    */
            /* Watch PORTA for action on Rotary Encoder or IR port                          */
            /* We want to catch the system timer value when we see action on the IR port    */
            /* Type A - Encoder is biphase input on two ports with                          */
            /*      Each detent click of the resolver being four changes in phase           */
            /*      Thus one complete cycle of gray code occurs per click                   */
            /*      Type A goes 11 -> 10 -> 00 -> 01 -> 11 and vice versa                   */
            /*      Type A goes 11 -> 01 -> 00 -> 10 -> 11 and vice versa                   */
            /*      This is the ALTRONICS device and my EBAY sourced devices                */
            case VOLUME_STATE_IDLE:
            {
                /* grab the time */
                Volume_Data.Timer_Time = TMR0L;
                Volume_Data.Timer_Time += (unsigned int) (TMR0H << 8);
                Current_PortA = PORTA & (PORT_BITS);                             

                if (Current_PortA != Temp_PortA)    /* then need to do something */
                {
                    /* get current ROT state */
                    Volume_Data.ROT_State = Current_PortA & (ROT_A + ROT_B);
                    /* get current IR state */
                    Volume_Data.IR_State = Current_PortA & (IR_IN);
                    
                    /* see if it is the rotary Encoder */
                    if (Volume_Data.ROT_State != Phase0)
                    {
                        /* CW 11 -> 10  - mid click remember */
                        /* CCW 11-> 01  - mid click remember */
                        if(Volume_Data.ROT_State_Prev  == Phase0)
                        {
                            __delay_us(ROTARY_ENC_SETTLE);   /* let things settle on initial switch */  
                            Volume_Data.Action = No_Action;
                        }
                        /* CW 11 -> 10 -> 00    i.e. phase 0 to phase 1*/ 
                        else if (Volume_Data.ROT_State_Prev  == Phase1)
                        {
                            if(Volume_Data.ROT_State == Phase2)
                            {
                                Volume_Data.Action = Volume_Up;
                            }
                            else
                            {
                                Volume_Data.Action = No_Action;                                
                            }
                        }
                        /* CW 11 -> 10 -> 00    i.e. phase 0 to phase 1*/ 
                        else if (Volume_Data.ROT_State_Prev  == Phase2)
                        {
                                Volume_Data.Action = No_Action;                                                            
                        }
                        /* CCW 11 -> 01-> 00    i.e. phase 0 to phase 3*/
                        else if (Volume_Data.ROT_State_Prev  == Phase3)
                        {
                            if(Volume_Data.ROT_State == Phase2)
                            {
                                Volume_Data.Action = Volume_Down;
                            }
                            else
                            {
                                Volume_Data.Action = No_Action;                                
                            }
                            
                        }
                    }

                    /************************************************/                    
                    // To decode IR..
                    // Watch IR input (remember it is inverted)
                    //
                    //State Machine:
                    // 
                    // State 1: Input low (1 received)
                    //  if goes high after a short time then the next character is a 1.  Wait until it goes low then back to state 1.
                    //  else if the output stays low after a short time but goes high after a long time, then the next character is a 0, wait until it goes high then go to state 0
                    //
                    // State 0: Input high (0 received)
                    //  if the output stays high after a short time, but goes low after a long time,  then the next character is a 1, Wait until it goes low then back to state 1.  
                    //  else if the output goes  low after a short time then the next character is a 0, wait until it goes high then go to state 0
                    //
                    //  event	ideal	min	max
                    //  short pulse	889?s	444?s	1333?s
                    //  short space	889?s	444?s	1333?s
                    //  long pulse	1778?s	1334?s	2222?s
                    //  long space	1778?s	1334?s	2222?s
                    // #define RC5_Short_Short 600
                    // #define RC5_Short_Long 1333
                    // #define RC5_Long_Short 1334
                    // #define RC5_Long_Long  2222
                    /************************************************/                    
                    else if (Volume_Data.IR_State != IR_IN)   /* IR low */
                    {
                        Volume_Data.TimeOut = 0;
                        // start at current time
                        Volume_Data.Timer_Time = TMR0L;
                        Volume_Data.Timer_Time += (unsigned int)(TMR0H << 8);
                        Volume_Data.Timer_Temp = Volume_Data.Timer_Time;
                        
                        Bit_Count = 1;  //us this as a bit counter on reading
                        Volume_Data.IR_RX = 1; /* always starts with a 1 */
                        Volume_Data.RC5_state = Out_1; /* always starts with a 1 */
 
                        do
                        {
                            /* get the time */
                            Volume_Data.Timer_Time = TMR0L;
                            Volume_Data.Timer_Time += (unsigned int)(TMR0H << 8);
                            Volume_Data.Interval = Volume_Data.Timer_Time - Volume_Data.Timer_Temp;                            
                            
                            switch ( Volume_Data.RC5_state )
                            {
                                /* "1" receive and RC5 initial state. */
                                /* high < short -> error */
                                /* high in short -> 1 */
                                /*      clear th high and come back to "1"*/
                                /* high in long -> 0 */
                                /*       goto "0"*/
                                /* still low > long -> error */
                                case Out_1:   //0x10
                                {
                                    if((PORTA & IR_IN) && (Volume_Data.Interval <= RC5_Short_Short))
                                    {
                                        Volume_Data.TimeOut = 1;
                                    } /* been too fast */
                                    else if ((PORTA & IR_IN) && (Volume_Data.Interval <= RC5_Short_Long))
                                    {
                                        Volume_Data.RC5_state = Mid_Out_1; /* next character is a 1 but need to wait until it returns low*/
                                        Volume_Data.Timer_Temp = Volume_Data.Timer_Time; /* reset timer marker */
                                        Volume_Data.IR_RX = 1 + (Volume_Data.IR_RX<<1);    /* shift one place and add 1 */
                                        Bit_Count++;  /* increment bits read */
                                    }
                                    else if ((PORTA & IR_IN) && (Volume_Data.Interval <= RC5_Long_Long))
                                    {
                                        Volume_Data.RC5_state = Out_0; /* next character is a 0*/
                                        Volume_Data.Timer_Temp = Volume_Data.Timer_Time; /* reset timer marker */
                                        Volume_Data.IR_RX =  Volume_Data.IR_RX<<1;    /* shift one place and add 0 */
                                        Bit_Count++;  /* increment bits read */
                                    }
                                    else if (Volume_Data.Interval >= RC5_Long_Long)
                                    {
                                        Volume_Data.TimeOut = 1;
                                    } /* been waiting too long */
                                    else /* not high yet */
                                    {
                                        
                                    };
                                        
                                }
                            break;
                            /* "1" and waiting till "0" */
                                case Mid_Out_1:  //0x20
                                {
                                    if((!(PORTA & IR_IN)) && (Volume_Data.Interval <= RC5_Short_Short))
                                    {
                                        Volume_Data.TimeOut = 1;
                                    } /* too short */
                                    else if ((!(PORTA & IR_IN)) && (Volume_Data.Interval <= RC5_Short_Long))
                                    {
                                        Volume_Data.Timer_Temp = Volume_Data.Timer_Time; /* reset timer marker */
                                        Volume_Data.RC5_state = Out_1;  /* next character is a 1 */
                                        
                                    } /* this is what we have been looking for*/
                                    else if (Volume_Data.Interval > RC5_Short_Long)
                                    {
                                        Volume_Data.TimeOut = 1;
                                    }; /* too long is bad */
                                }
                            break;
                            
                    //  if the output stays high after a short time, but goes low after a long time,  then the next character is a 1, Wait until it goes low then back to state 1.  
                    //  else if the output goes  low after a short time then the next character is a 0, wait until it goes high then go to state 0
                            
                                /* "0" received. */
                                /* low < short -> error */
                                /* low in short -> 0 */
                                /*      clear the low and come back to "0"*/
                                /* low in long -> 1 */
                                /*       goto "1"*/
                                /* still high > long -> error */
                                case Out_0:    //0x30
                                {
                                    if(!(PORTA & IR_IN) && (Volume_Data.Interval <= RC5_Short_Short))
                                    {
                                        Volume_Data.TimeOut = 1;
                                    } /* been too fast */
                                    else if (!(PORTA & IR_IN) && (Volume_Data.Interval <= RC5_Short_Long))
                                    {
                                        Volume_Data.RC5_state = Mid_Out_0; /* next character is a 0 but need to clear the low*/
                                        Volume_Data.Timer_Temp = Volume_Data.Timer_Time; /* reset timer marker */
                                        Volume_Data.IR_RX =  Volume_Data.IR_RX<<1;    /* shift one place and add 0 */
                                        Bit_Count++;  /* increment bits read */
                                    }
                                    else if (!(PORTA & IR_IN) && (Volume_Data.Interval <= RC5_Long_Long))
                                    {
                                        Volume_Data.RC5_state = Out_1; /* next character is a 1*/
                                        Volume_Data.Timer_Temp = Volume_Data.Timer_Time; /* reset timer marker */
                                        Volume_Data.IR_RX = 1 + (Volume_Data.IR_RX<<1);    /* shift one place and add 1 */
                                        Bit_Count++;  /* increment bits read */
                                    }
                                    else if (Volume_Data.Interval >= RC5_Long_Long)
                                    {
                                        Volume_Data.TimeOut = 1;
                                    } /* been waiting too long */
                                }
                            break;
                            /* "1" and waiting till "0" */
                                case Mid_Out_0:  //0x40
                                {
                                    if((PORTA & IR_IN) && (Volume_Data.Interval <= RC5_Short_Short))
                                    {
                                        Volume_Data.TimeOut = 1;
                                    } /* too short */
                                    else if ((PORTA & IR_IN) && (Volume_Data.Interval <= RC5_Short_Long))
                                    {
                                        Volume_Data.Timer_Temp = Volume_Data.Timer_Time; /* reset timer marker */
                                        Volume_Data.RC5_state = Out_0;  /* next character is a 1 */
                                     } /* this is what we have been looking for*/
                                    else if (Volume_Data.Interval > RC5_Short_Long)
                                    {
                                        Volume_Data.TimeOut = 1;
                                    }; /* too long is bad */
                                }
                            break;
                            default:
                                {
                                    Volume_Data.TimeOut = 1;
                                }
                            } 
                            
                        }    while ((!Volume_Data.TimeOut) && (Bit_Count < RC5_Bits));

                                                
                        // I2C_Volume_Word((unsigned char) (Volume_Data.Toggle));
                        Volume_Data.Toggle = (Volume_Data.IR_RX & 0x800)>>11;
                        /* now read the Address Field                           */
                        /* this is five bits, which will be found in bits       */
                        /* 6-10 of the IR Rx data                               */
                        /* need to mask the toggle bit                          */
                        Volume_Data.Address = ((Volume_Data.IR_RX & 0x7C)>>5);
                        /* now read the command Field                           */
                        /* this is six bits, which will be found in bits        */
                        /* 0-6 of the IR Rx data                                */
                        /* need to mask the toggle bit and command              */
                        Volume_Data.Command = ((Volume_Data.IR_RX & 0x3F));

//             I2C_Volume_Word((unsigned char) Volume_Data.Toggle);
//             I2C_Volume_Word((unsigned char) Volume_Data.Address);
//             I2C_Volume_Word((unsigned char) Volume_Data.Command);

                        
                        if ((Bit_Count == RC5_Bits)&&(Volume_Data.Address == Volume_Data.IR_Addr ) && (Volume_Data.Command == Volume_Down_IR)) Volume_Data.Action = Volume_Down;
                        else if ((Bit_Count == RC5_Bits)&&(Volume_Data.Address == Volume_Data.IR_Addr ) && (Volume_Data.Command == Volume_Up_IR)) Volume_Data.Action = Volume_Up;
                        else if ((Bit_Count == RC5_Bits)&&(Volume_Data.Address == Volume_Data.IR_Addr ) && (Volume_Data.Command == Volume_Mute_IR)) 
                        {
                            Volume_Data.Action = Volume_Mute;
                            if(Volume_Data.Mute == 0)
                            {
                                Volume_Data.Mute = 1;
                            }
                            else
                            {
                                Volume_Data.Mute = 0;
//                                I2C_Volume_Word((unsigned char) Volume_Data.Volume);
                            }
                        }
                        else Volume_Data.Action = No_Action;
                        Current_PortA = PORTA & (PORT_BITS);  
                    }
                }
                
                if(Volume_Data.Action == Volume_Up)
                {
                    if(Volume_Data.Volume <= (VOLUME_VAL_UPPER - Volume_Data.Increment))
                        Volume_Data.Volume += Volume_Data.Increment;
                    else
                        Volume_Data.Volume = VOLUME_VAL_UPPER;

                    I2C_Volume_Word((unsigned char) Volume_Data.Volume);
                   ASSERT_CS;   /* use CS as line to drive a LED indication activity */
                   __delay_ms(Volume_Delay_ms);
                   DEASSERT_CS; /* now turn it off again! */
                   Volume_Data.Data_To_Save = 1;   /* remind ourselves there is data to save */
                   Volume_Data.Save_Timer  = Save_Timer_Val;    /* reset save timer */
                   Volume_Data.Slow_Loops += 1;    /* This counts how many times we have incremented */
                   Volume_Data.Fast_Loops = 0;     /* Reset Fast Loop counter, increment slow while this has not gone off */
                   Volume_Data.Action = No_Action;
                }
                else if (Volume_Data.Action == Volume_Down)
                {
                    if(Volume_Data.Volume >= (VOLUME_VAL_LOWER + Volume_Data.Increment))
                        Volume_Data.Volume -= Volume_Data.Increment;
                    else
                        Volume_Data.Volume = VOLUME_VAL_LOWER;

                    I2C_Volume_Word((unsigned char) Volume_Data.Volume);
                   ASSERT_CS;   /* use CS as line to drive a LED indication activity */
                   __delay_ms(Volume_Delay_ms);
                   DEASSERT_CS; /* now turn it off again! */
                   Volume_Data.Data_To_Save = 1;   /* remind ourselves there is data to save */
                   Volume_Data.Save_Timer  = Save_Timer_Val;    /* reset save timer */
                   Volume_Data.Slow_Loops += 1;   /* This counts how many times we have incremented */
                   Volume_Data.Fast_Loops = 0;     /* Reset Fast Loop counter, increment slow while this has not gone off */
                   Volume_Data.Action = No_Action;
                }
                else if (Volume_Data.Action == Volume_Mute)
                {
                    if (Volume_Data.Mute == 1)
                    {
                        I2C_Volume_Word((unsigned char) VOLUME_VAL_LOWER);
                        ASSERT_CS;   /* use CS as line to drive a LED indication activity */
                        __delay_ms(mute_delay);
                        DEASSERT_CS; /* now turn it off again! */
                        Volume_Data.Data_To_Save = 0;   /* remind ourselves there is data to save */
                        Volume_Data.Fast_Loops += 1; 
                    }
                    else if  (Volume_Data.Mute == 0)
                    {
                        I2C_Volume_Word((unsigned char) Volume_Data.Volume);
                        ASSERT_CS;   /* use CS as line to drive a LED indication activity */
                        __delay_ms(mute_delay);
                        DEASSERT_CS; /* now turn it off again! */
                        Volume_Data.Data_To_Save = 0;   /* remind ourselves there is data to save */
                        Volume_Data.Fast_Loops += 1; 
                    }
                    Volume_Data.Action = No_Action;
                }
                else
                {
//                    I2C_Volume_Word((unsigned char) (0x00));
                    Volume_Data.Action = No_Action;
                    Volume_Data.Fast_Loops += 1; 
                }

                /* check slow / fast count and set speed */
                if (Volume_Data.Slow_Loops > Slow_Fast_Count)
                    Volume_Data.Increment = Fast_Steps;
                if (Volume_Data.Fast_Loops >= Fast_Slow_Count)
                {
                    Volume_Data.Slow_Loops = 0;
                    Volume_Data.Increment = Slow_Steps;
                }

                /* check if we need to save to memory */
                if ((Volume_Data.Data_To_Save == 1) && (Volume_Data.Save_Timer))
                {
                    Volume_Data.Save_Timer  = Volume_Data.Save_Timer-1;    /* decrement timer */
                    if (Volume_Data.Save_Timer == 0)
                    {
                       I2C_Volume_Word((unsigned char) Volume_Data.Volume);
                       Volume_Data.Data_To_Save = 0; /* we have saved now */
                       SaveVolToFlash();
                    }
                }
                
                /* save the current state as previous */
                Volume_Data.ROT_State_Prev  = Volume_Data.ROT_State;
                Volume_Data.IR_State_Prev   = Volume_Data.IR_State;
                Temp_PortA = Current_PortA;  
            }
            break;
        }
        
    } while(1); // for ever


//    do {
//        
//        I2C_Volume_Word((unsigned char) (TMR0L));
//        I2C_Volume_Word((unsigned char) (TMR0H));
//        __delay_us(100);        
//    } while(1);
    
    return;
}
